import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:line_icons/line_icons.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/models/article.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/services/bookmark_service.dart';
import 'package:wordpress_app/utils/cached_image.dart';
import 'package:wordpress_app/utils/next_screen.dart';
import 'package:wordpress_app/widgets/category_chip.dart';
import 'package:wordpress_app/widgets/video_icon.dart';

class BookmarkCard extends ConsumerWidget {
  final Article article;
  const BookmarkCard({super.key, required this.article});

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final String heroTag = 'bookmark_${article.id}';
    final configs = ref.read(configBlocProvider).configs!;

    return InkWell(
        child: Container(
            decoration: BoxDecoration(borderRadius: BorderRadius.circular(5)),
            child: Row(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Stack(
                  children: [
                    SizedBox(
                      height: 110,
                      width: 110,
                      child: Hero(
                          tag: heroTag,
                          child: CustomCacheImage(
                              imageUrl: article.image, radius: 5)),
                    ),
                    Positioned.fill(
                      child: Align(
                        alignment: Alignment.center,
                        child: VideoIcon(article: article, iconSize: 40),
                      ),
                    ),
                    Positioned(
                      top: 4,
                      right: 4,
                      child: Material(
                        color: Theme.of(context)
                            .colorScheme
                            .surface
                            .withValues(alpha: 0.9),
                        shape: const CircleBorder(),
                        child: InkWell(
                          onTap: () =>
                              BookmarkService().removeFromBookmarkList(article),
                          customBorder: const CircleBorder(),
                          child: Padding(
                            padding: const EdgeInsets.all(6),
                            child: Icon(
                              LineIcons.trash,
                              size: 16,
                              color: Theme.of(context).colorScheme.onSurface,
                            ),
                          ),
                        ),
                      ),
                    ),
                  ],
                ),
                const SizedBox(
                  width: 15,
                ),
                Expanded(
                  child: Column(
                    mainAxisAlignment: MainAxisAlignment.start,
                    crossAxisAlignment: CrossAxisAlignment.start,
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      CategoryChip(article: article),
                      const SizedBox(height: 8),
                      Flexible(
                        child: Text(
                          AppService.getNormalText(article.title!),
                          maxLines: 3,
                          overflow: TextOverflow.ellipsis,
                          style: Theme.of(context)
                              .textTheme
                              .headlineSmall
                              ?.copyWith(
                                  fontSize: 17, fontWeight: FontWeight.w600),
                        ),
                      ),
                    ],
                  ),
                ),
              ],
            )),
        onTap: () =>
            navigateToDetailsScreen(context, article, heroTag, configs));
  }
}
